"""
Parent-Teacher Meeting (PTM) Web Application
Complete Flask Backend with SQLite Database
"""

import os
import csv
import sqlite3
import smtplib
from email.message import EmailMessage
from datetime import datetime
from functools import wraps
from flask import Flask, render_template, request, redirect, url_for, session

app = Flask(__name__)
app.secret_key = 'ptm-secret-key-2024'

# ========================
# CONFIGURATION
# ========================
CSV_FILE = 'students.csv'
DATABASE = 'feedback.db'
VOICE_FOLDER = 'static/voices'

SMTP_SERVER = "mail.diet-icpinaka26.tech"
SMTP_PORT = 465
SMTP_EMAIL = "ptmcsdm@diet-icpinaka26.tech"
SMTP_PASSWORD = "Diet@Pinaka"
ADMIN_EMAIL = "rapetanaidu478@gmail.com"

ADMIN_USERNAME = "Dietptm"
ADMIN_PASSWORD = "Dietptm"

os.makedirs(VOICE_FOLDER, exist_ok=True)

# ========================
# DATABASE INIT
# ========================
def init_db():
    conn = sqlite3.connect(DATABASE)
    cursor = conn.cursor()
    cursor.execute("""
        CREATE TABLE IF NOT EXISTS parent_feedback (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            roll TEXT,
            name TEXT,
            rating INTEGER,
            text_feedback TEXT,
            voice_file TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    """)
    conn.commit()
    conn.close()

init_db()

# ========================
# LOAD STUDENTS
# ========================
def load_students():
    students = {}
    with open(CSV_FILE, encoding="utf-8-sig") as f:
        reader = csv.DictReader(f)
        for row in reader:
            roll = row.get("Roll. No", "").strip().upper()
            if roll:
                students[roll] = {
                    "roll": roll,
                    "name": row.get("Name of the Student", "").strip(),
                    "contact": row.get("Contact Number", "").strip(),
                    "attendance": row.get("Attendance", "").strip(),
                    "sgpa": row.get("SGPA", "").strip(),
                    "cgpa": row.get("Total CGPA", "").strip(),
                    "backlogs": row.get("Backlogs till now", "").strip(),
                    "mentor_mobile": row.get("Mentor mobile Number", "").strip()
                }
    return students

STUDENTS = load_students()

# ========================
# HELPERS
# ========================
def get_attendance_status(att):
    try:
        val = float(att.replace("%", ""))
        if val >= 75:
            return "good", "Good Attendance", val
        elif val >= 65:
            return "warning", "Needs Attention", val
        else:
            return "danger", "Detention Risk!", val
    except:
        return "unknown", "Not Available", 0

def get_backlog_status(b):
    try:
        import re
        nums = re.findall(r"\d+", str(b))
        count = int(nums[0]) if nums else 0
        if count == 0:
            return "good", "No Backlogs", 0
        elif count <= 2:
            return "warning", "Needs Improvement", count
        else:
            return "danger", "Critical Academic Risk", count
    except:
        return "unknown", "Not Available", 0

# ========================
# EMAIL (WORKING)
# ========================
def send_email_notification(roll, name, rating, text, voice_file):
    try:
        msg = EmailMessage()
        msg["Subject"] = f"PTM Feedback – {roll}"
        msg["From"] = SMTP_EMAIL
        msg["To"] = ADMIN_EMAIL

        msg.set_content(f"""
PTM Feedback Received

Name : {name}
Roll : {roll}
Rating : {rating}

{text or "No text feedback"}
""")

        if voice_file and os.path.exists(voice_file):
            with open(voice_file, "rb") as f:
                msg.add_attachment(
                    f.read(),
                    maintype="audio",
                    subtype="webm",
                    filename=os.path.basename(voice_file)
                )

        server = smtplib.SMTP_SSL(SMTP_SERVER, SMTP_PORT)
        server.login(SMTP_EMAIL, SMTP_PASSWORD)
        server.send_message(msg)
        server.quit()

        print("MAIL SENT:", roll)

    except Exception as e:
        print("MAIL ERROR:", e)

# ========================
# AUTH DECORATORS
# ========================
def login_required(f):
    @wraps(f)
    def wrap(*args, **kwargs):
        if "student_roll" not in session:
            return redirect(url_for("login"))
        return f(*args, **kwargs)
    return wrap

def admin_required(f):
    @wraps(f)
    def wrap(*args, **kwargs):
        if "admin_logged_in" not in session:
            return redirect(url_for("admin_login"))
        return f(*args, **kwargs)
    return wrap

# ========================
# ROUTES – PARENT
# ========================
@app.route("/")
def index():
    return redirect(url_for("login"))

@app.route("/login", methods=["GET", "POST"])
def login():
    if request.method == "POST":
        roll = request.form["roll"].upper()
        password = request.form["password"]
        language = request.form.get("language", "en")

        if roll in STUDENTS and STUDENTS[roll]["contact"] == password:
            session["student_roll"] = roll
            session["language"] = language
            return redirect(url_for("dashboard"))

    return render_template("login.html")

@app.route("/dashboard")
@login_required
def dashboard():
    roll = session["student_roll"]
    language = session.get("language", "en")
    student = STUDENTS[roll]

    att_status, att_message, att_value = get_attendance_status(student["attendance"])
    bl_status, bl_message, bl_count = get_backlog_status(student["backlogs"])

    return render_template(
        "dashboard.html",
        student=student,
        language=language,
        att_status=att_status,
        att_message=att_message,
        att_value=att_value,
        backlog_status=bl_status,
        backlog_message=bl_message,
        backlog_count=bl_count
    )

@app.route("/feedback", methods=["GET", "POST"])
@login_required
def feedback():
    roll = session["student_roll"]
    student = STUDENTS[roll]
    language = session.get("language", "en")

    if request.method == "POST":
        rating = int(request.form.get("rating", 0))
        text = request.form.get("text_feedback", "").strip()

        voice_filename = None
        if "voice_file" in request.files:
            voice = request.files["voice_file"]
            if voice.filename:
                voice_filename = f"{roll}_{datetime.now().strftime('%Y%m%d_%H%M%S')}.webm"
                voice.save(os.path.join(VOICE_FOLDER, voice_filename))

        conn = sqlite3.connect(DATABASE)
        cursor = conn.cursor()
        cursor.execute("""
            INSERT INTO parent_feedback
            (roll, name, rating, text_feedback, voice_file)
            VALUES (?, ?, ?, ?, ?)
        """, (roll, student["name"], rating, text, voice_filename))
        conn.commit()
        conn.close()

        voice_path = os.path.join(VOICE_FOLDER, voice_filename) if voice_filename else None
        send_email_notification(roll, student["name"], rating, text, voice_path)

        return render_template("thank_you.html", language=language)

    return render_template("feedback.html", student=student, language=language)

@app.route("/logout")
def logout():
    session.clear()
    return redirect(url_for("login"))

# ========================
# ROUTES – ADMIN
# ========================
@app.route("/admin", methods=["GET", "POST"])
def admin_login():
    if request.method == "POST":
        if (request.form["username"] == ADMIN_USERNAME and
            request.form["password"] == ADMIN_PASSWORD):
            session["admin_logged_in"] = True
            return redirect(url_for("admin_dashboard"))

    return render_template("admin_login.html")

@app.route("/admin/dashboard")
@admin_required
def admin_dashboard():
    conn = sqlite3.connect(DATABASE)
    conn.row_factory = sqlite3.Row
    cursor = conn.cursor()
    cursor.execute("SELECT * FROM parent_feedback ORDER BY created_at DESC")
    feedbacks = cursor.fetchall()
    conn.close()

    return render_template("admin_dashboard.html", feedbacks=feedbacks)

@app.route("/admin/logout")
def admin_logout():
    session.pop("admin_logged_in", None)
    return redirect(url_for("admin_login"))

# ========================
# RUN
# ========================
if __name__ == "__main__":
    print("PTM APPLICATION RUNNING")
    app.run(debug=False, host="0.0.0.0", port=5000)
